<?php

/**
 * nextbit functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package nextbit
 */

if ( !function_exists( 'nextbit_setup' ) ):
    /**
     * Sets up theme defaults and registers support for various WordPress features.
     *
     * Note that this function is hooked into the after_setup_theme hook, which
     * runs before the init hook. The init hook is too late for some features, such
     * as indicating support for post thumbnails.
     */
    function nextbit_setup() {
        /*
         * Make theme available for translation.
         * Translations can be filed in the /languages/ directory.
         * If you're building a theme based on nextbit, use a find and replace
         * to change 'nextbit' to the name of your theme in all the template files.
         */
        load_theme_textdomain( 'nextbit', get_template_directory() . '/languages' );

        // Add default posts and comments RSS feed links to head.
        add_theme_support( 'automatic-feed-links' );

        /*
         * Let WordPress manage the document title.
         * By adding theme support, we declare that this theme does not use a
         * hard-coded <title> tag in the document head, and expect WordPress to
         * provide it for us.
         */
        add_theme_support( 'title-tag' );

        /*
         * Enable support for Post Thumbnails on posts and pages.
         *
         * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
         */
        add_theme_support( 'post-thumbnails' );

        // This theme uses wp_nav_menu() in one location.
        register_nav_menus( [
            'main-menu' => esc_html__( 'Main Menu', 'nextbit' ),
        ] );

        /*
         * Switch default core markup for search form, comment form, and comments
         * to output valid HTML5.
         */
        add_theme_support( 'html5', [
            'search-form',
            'comment-form',
            'comment-list',
            'gallery',
            'caption',
        ] );

        // Set up the WordPress core custom background feature.
        add_theme_support( 'custom-background', apply_filters( 'nextbit_custom_background_args', [
            'default-color' => 'ffffff',
            'default-image' => '',
        ] ) );

        // Add theme support for selective refresh for widgets.
        add_theme_support( 'customize-selective-refresh-widgets' );

        //Enable custom header
        add_theme_support( 'custom-header' );

        /**
         * Add support for core custom logo.
         *
         * @link https://codex.wordpress.org/Theme_Logo
         */
        add_theme_support( 'custom-logo', [
            'height'      => 250,
            'width'       => 250,
            'flex-width'  => true,
            'flex-height' => true,
        ] );

        /**
         * Enable suporrt for Post Formats
         *
         * see: https://codex.wordpress.org/Post_Formats
         */
        add_theme_support( 'post-formats', [
            'image',
            'audio',
            'video',
            'gallery',
            'quote'
        ] );

        // Add theme support for selective refresh for widgets.
        //add_theme_support( 'customize-selective-refresh-widgets' );

        // enable woocommerce
        add_theme_support('woocommerce');

        // Add support for Block Styles.
        add_theme_support( 'wp-block-styles' );

        // Add support for full and wide align images.
        add_theme_support( 'align-wide' );

        // Add support for editor styles.
        add_theme_support( 'editor-styles' );

        // Add support for responsive embedded content.
        add_theme_support( 'responsive-embeds' );

        // remove block widget support
        remove_theme_support( 'widgets-block-editor' );

        add_image_size( 'nextbit-case-details', 1170, 600, ['center', 'center'] );
        add_image_size( 'nextbit-post-thumb', 500, 350, ['center', 'center'] );
        add_image_size( 'nextbit-case-thumb', 700, 544, ['center', 'center'] );
    }
endif;
add_action( 'after_setup_theme', 'nextbit_setup' );

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function nextbit_content_width() {
    // This variable is intended to be overruled from themes.
    // Open WPCS issue: {@link https://github.com/WordPress-Coding-Standards/WordPress-Coding-Standards/issues/1043}.
    // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound
    $GLOBALS['content_width'] = apply_filters( 'nextbit_content_width', 640 );
}
add_action( 'after_setup_theme', 'nextbit_content_width', 0 );

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function nextbit_widgets_init() {
    $movie_sidebar_switch = get_theme_mod( 'movie_sidebar_switch', false );
    $service_sidebar_switch = get_theme_mod( 'service_sidebar_switch', false );
    /**
     * blog sidebar
     */
    register_sidebar( [
        'name'          => esc_html__( 'Blog Sidebar', 'nextbit' ),
        'id'            => 'blog-sidebar',
        'before_widget' => '<div id="%1$s" class="sidebar-widget widget %2$s"><div class="widget-content">',
        'after_widget'  => '</div></div>',
        'before_title'  => '<h5 class="sidebar-title">',
        'after_title'   => '</h5>',
    ] );

    if( $movie_sidebar_switch == true ) {
        register_sidebar( [
            'name'          => esc_html__( 'Movie Sidebar', 'nextbit' ),
            'id'            => 'movie-sidebar',
            'before_widget' => '<div id="%1$s" class="sidebar-widget widget %2$s"><div class="widget-content">',
            'after_widget'  => '</div></div>',
            'before_title'  => '<h5 class="sidebar-title">',
            'after_title'   => '</h5>',
        ] );
    }

    if( $service_sidebar_switch == true ) {
        register_sidebar( [
            'name'          => esc_html__( 'Service Sidebar', 'nextbit' ),
            'id'            => 'service-sidebar',
            'before_widget' => '<div id="%1$s" class="sidebar-widget widget %2$s"><div class="widget-content">',
            'after_widget'  => '</div></div>',
            'before_title'  => '<h5 class="sidebar-title">',
            'after_title'   => '</h5>',
        ] );
    }

    // footer default
    for ( $num = 1; $num <= 3; $num++ ) {
        register_sidebar( [
            'name'          => sprintf( esc_html__( 'Footer %1$s', 'nextbit' ), $num ),
            'id'            => 'footer-' . $num,
            'description'   => sprintf( esc_html__( 'Footer %1$s', 'nextbit' ), $num ),
            'before_widget' => '<div id="%1$s" class="footer_widget widget mt-30 %2$s widget-%1$s">',
            'after_widget'  => '</div>',
            'before_title'  => '<h3 class="footer_widget_title text-uppercase">',
            'after_title'   => '</h3>',
        ] );
    }
    register_sidebar( [
        'name'          => esc_html__( 'Footer 4', 'nextbit' ),
        'id'            => 'footer-4',
        'description'   => esc_html__( 'Footer 4', 'nextbit' ),
    ] );
    
    register_sidebar( [
        'name'          => esc_html__( 'Footer 5', 'nextbit' ),
        'id'            => 'footer-5',
        'description'   => esc_html__( 'Footer 5', 'nextbit' ),
    ] );

    register_sidebar(array(
		'name'          => esc_html__('Footer 6', 'nextbit'),
		'id'            => 'footer-6',
		'description'   => esc_html__('Add footer widget here.', 'nextbit'),
		'before_widget' => '<section id="%1$s" class="widget footer__link %2$s">',
		'after_widget'  => '</section>',
		'before_title'  => '<h3 class="widget-title">',
		'after_title'   => '</h3>',
	));
    register_sidebar(array(
		'name'          => esc_html__('Footer 7', 'nextbit'),
		'id'            => 'footer-7',
		'description'   => esc_html__('Add footer widget here.', 'nextbit'),
		'before_widget' => '<section id="%1$s" class="widget footer__link %2$s">',
		'after_widget'  => '</section>',
		'before_title'  => '<h3 class="widget-title">',
		'after_title'   => '</h3>',
	));
    register_sidebar(array(
		'name'          => esc_html__('Recent 8', 'nextbit'),
		'id'            => 'footer-8',
		'description'   => esc_html__('Add footer widget here.', 'nextbit'),
		'before_widget' => '<section id="%1$s" class="widget footer__widget-post %2$s">',
		'after_widget'  => '</section>',
		'before_title'  => '<h3 class="widget-title">',
		'after_title'   => '</h3>',
	));
    
}
add_action( 'widgets_init', 'nextbit_widgets_init' );

// custom admin script
function nextbit_admin_scripts() {
    wp_enqueue_media();
    wp_enqueue_script( 'adminCustom', NEXTBIT_THEME_JS_DIR . 'custom-admin.js', ['jquery'], false, true );
}
add_action( 'admin_enqueue_scripts', 'nextbit_admin_scripts' );

/**
 * Enqueue scripts and styles.
 * @since 1.0.0
 */

define( 'NEXTBIT_THEME_DIR', get_template_directory() );
define( 'NEXTBIT_THEME_URI', get_template_directory_uri() );
define( 'NEXTBIT_THEME_CSS_DIR', NEXTBIT_THEME_URI . '/assets/css/' );
define( 'NEXTBIT_THEME_JS_DIR', NEXTBIT_THEME_URI . '/assets/js/' );
define( 'NEXTBIT_THEME_INC', NEXTBIT_THEME_DIR . '/inc/' );

if (!defined('NEXTBIT_VARIATION_GALLERY_ACTIVED')) {
    define('NEXTBIT_VARIATION_GALLERY_ACTIVED', in_array('woo-variation-gallery/woo-variation-gallery.php', apply_filters('active_plugins', get_option('active_plugins'))));
}


/**
 * nextbit_scripts description
 * @return [type] [description]
 */
function nextbit_scripts() {

    wp_enqueue_style( 'nextbit-fonts', nextbit_fonts_url(), [], null );
    wp_enqueue_style( 'bootstrap', NEXTBIT_THEME_CSS_DIR . 'bootstrap.css', [] );
    wp_enqueue_style( 'owl', NEXTBIT_THEME_CSS_DIR . 'owl.css', [] );
    wp_enqueue_style( 'animate', NEXTBIT_THEME_CSS_DIR . 'animate.css', [] );
    wp_enqueue_style( 'custom-animate', NEXTBIT_THEME_CSS_DIR . 'custom-animate.css', [] );
    wp_enqueue_style( 'fontawesome-font', NEXTBIT_THEME_CSS_DIR . 'fontawesome.min.css', [] );
    wp_enqueue_style( 'footer', NEXTBIT_THEME_CSS_DIR . 'footer.css', [] );
    wp_enqueue_style( 'global', NEXTBIT_THEME_CSS_DIR . 'global.css', [] );
    wp_enqueue_style( 'header', NEXTBIT_THEME_CSS_DIR . 'header.css', [] );
    wp_enqueue_style( 'swiper', NEXTBIT_THEME_CSS_DIR . 'swiper.css', [] );
    wp_enqueue_style( 'hover', NEXTBIT_THEME_CSS_DIR . 'hover.css', [] );
    wp_enqueue_style( 'icofont', NEXTBIT_THEME_CSS_DIR . 'icofont.css', [] );
    wp_enqueue_style( 'flaticon', NEXTBIT_THEME_CSS_DIR . 'flaticon.css', [] );
    wp_enqueue_style( 'jquery-bootstrap-touchspin', NEXTBIT_THEME_CSS_DIR . 'jquery.bootstrap-touchspin.css', [] );
    wp_enqueue_style( 'fancybox', NEXTBIT_THEME_CSS_DIR . 'jquery.fancybox.min.css', [] );
    wp_enqueue_style( 'mCustomScrollbar', NEXTBIT_THEME_CSS_DIR . 'jquery.mCustomScrollbar.min.css', [] );
    wp_enqueue_style( 'jquery-ui', NEXTBIT_THEME_CSS_DIR . 'jquery-ui.css', [] );
    wp_enqueue_style( 'linearicons', NEXTBIT_THEME_CSS_DIR . 'linearicons.css', [] );
    wp_enqueue_style( 'bootstrap', NEXTBIT_THEME_CSS_DIR . 'bootstrap.css', [] );
    wp_enqueue_style( 'lightcase', NEXTBIT_THEME_CSS_DIR . 'lightcase.css', [] );
    wp_enqueue_style( 'magnific-popup', NEXTBIT_THEME_CSS_DIR . 'magnific-popup.css', [] );
    wp_enqueue_style( 'odometer', NEXTBIT_THEME_CSS_DIR . 'odometer.css', [] );
    wp_enqueue_style( 'metisMenu', NEXTBIT_THEME_CSS_DIR . 'metisMenu.css', [] );
    wp_enqueue_style( 'nextbit-core', NEXTBIT_THEME_CSS_DIR . 'nextbit-core.css', [] );
    wp_enqueue_style( 'responsive', NEXTBIT_THEME_CSS_DIR . 'responsive.css', [] );
    wp_enqueue_style( 'resposive-two', NEXTBIT_THEME_CSS_DIR . 'resposive-two.css', [] );
    wp_enqueue_style( 'nextbit-companion', NEXTBIT_THEME_CSS_DIR . 'nextbit-companion.css', [] );
    wp_enqueue_style( 'nextbit-variation', NEXTBIT_THEME_CSS_DIR . 'nextbit-variation.css', [] );
    wp_enqueue_style( 'nextbit-custom', NEXTBIT_THEME_CSS_DIR . 'nextbit-custom.css', [] );
    wp_enqueue_style( 'internet-provider', NEXTBIT_THEME_CSS_DIR . 'internet-provider.css', [] );

    wp_enqueue_style( 'nextbit-woocommerce', NEXTBIT_THEME_CSS_DIR . 'nextbit-woocommerce.css', [] );
    if ( NEXTBIT_VARIATION_GALLERY_ACTIVED ) {
        wp_enqueue_style( 'nextbit-variation', NEXTBIT_THEME_CSS_DIR . 'nextbit-woo-variation-gallery.css', [] );
    }
    wp_enqueue_style( 'nextbit-custom', NEXTBIT_THEME_CSS_DIR . 'nextbit-custom.css', [] );
    wp_enqueue_style( 'nextbit-style', get_stylesheet_uri() );

    $my_current_lang = apply_filters( 'wpml_current_language', NULL );

    // rtl css files
    $rtl_enable = get_theme_mod( 'rtl_switch', false );
    if ( $my_current_lang != 'en' && $rtl_enable ) {
        wp_enqueue_style( 'nextbit-rtl', NEXTBIT_THEME_CSS_DIR . 'rtl.css', [] );
        wp_enqueue_style( 'nextbit-responsive-rtl', NEXTBIT_THEME_CSS_DIR . 'responsive-rtl.css', [] );
    }

    // all js files
    wp_enqueue_script( 'appear', NEXTBIT_THEME_JS_DIR . 'appear.js', ['jquery'], false, true );
    wp_enqueue_script( 'nextbit-popper', NEXTBIT_THEME_JS_DIR . 'popper-min.js', ['jquery'], false, true );
    wp_enqueue_script( 'bootstrap', NEXTBIT_THEME_JS_DIR . 'bootstrap.min.js', ['jquery'], false, true );
    wp_enqueue_script( 'custom-admin', NEXTBIT_THEME_JS_DIR . 'custom-admin.js', ['jquery'], false, true );
    wp_enqueue_script( 'easing', NEXTBIT_THEME_JS_DIR . 'jquery.easing.min.js', ['jquery'], false, true );
    wp_enqueue_script( 'fancybox', NEXTBIT_THEME_JS_DIR . 'jquery.fancybox.js', ['jquery'], false, true );
    wp_enqueue_script( 'mCustomScrollbar', NEXTBIT_THEME_JS_DIR . 'jquery.mCustomScrollbar.concat.min.js', ['jquery'], false, true );
    wp_enqueue_script( 'mixitup', NEXTBIT_THEME_JS_DIR . 'jquery.mixitup.js', ['jquery'], false, true );
    wp_enqueue_script( 'paroller', NEXTBIT_THEME_JS_DIR . 'jquery.paroller.min.js', ['jquery'], false, true );
    wp_enqueue_script( 'mixitup', NEXTBIT_THEME_JS_DIR . 'mixitup.js', ['jquery'], false, true );
    wp_enqueue_script( 'typeit', NEXTBIT_THEME_JS_DIR . 'typeit.js', ['jquery'], false, true );
    wp_enqueue_script( 'magnific-popup', NEXTBIT_THEME_JS_DIR . 'jquery.magnific-popup.min.js', ['jquery'], false, true );
    wp_enqueue_script( 'odometer', NEXTBIT_THEME_JS_DIR . 'odometer.min.js', ['jquery'], false, true );
    wp_enqueue_script( 'metisMenu', NEXTBIT_THEME_JS_DIR . 'metisMenu.min.js', ['jquery'], false, true );
    wp_enqueue_script( 'nav-tool', NEXTBIT_THEME_JS_DIR . 'nav-tool.js', ['jquery'], false, true );
    wp_enqueue_script( 'owl', NEXTBIT_THEME_JS_DIR . 'owl.js', ['jquery'], false, true );
    wp_enqueue_script( 'swiper-min', NEXTBIT_THEME_JS_DIR . 'swiper.min.js', ['jquery'], false, true );
    wp_enqueue_script( 'lightcase', NEXTBIT_THEME_JS_DIR . 'lightcase.js', ['jquery'], false, true );
    wp_enqueue_script( 'backToTop', NEXTBIT_THEME_JS_DIR . 'backToTop.js', ['jquery'], false, true );
    wp_enqueue_script( 'jquery-ui-core', ['jquery'], false, true );
    wp_enqueue_script( 'parallax', NEXTBIT_THEME_JS_DIR . 'parallax.min.js', ['jquery'], false, true );
    wp_enqueue_script( 'cursor-bundle', NEXTBIT_THEME_JS_DIR . 'cursor-bundle.js', ['jquery'], false, true );
    wp_enqueue_script( 'counterup', NEXTBIT_THEME_JS_DIR . 'jquery.counterup.min.js', ['jquery'], false, true );
    wp_enqueue_script( 'waypoints', NEXTBIT_THEME_JS_DIR . 'waypoints.min.js', ['jquery'], false, true );
    wp_enqueue_script( 'tilt-jquery', NEXTBIT_THEME_JS_DIR . 'tilt.jquery.min.js', ['jquery'], false, true );
    wp_enqueue_script( 'wow', NEXTBIT_THEME_JS_DIR . 'wow.js', ['jquery'], false, true );
    wp_enqueue_script( 'nextbit-yith', NEXTBIT_THEME_JS_DIR . 'nextbit-yith.js', ['jquery'], false, true );

    // rtl js files
    if ( $my_current_lang != 'en' && $rtl_enable ) {
        wp_enqueue_script( 'nextbit-rtl-main', NEXTBIT_THEME_JS_DIR . 'rtl-main.js', ['jquery'], false, true );
    } else {
        wp_enqueue_script( 'nextbit-main', NEXTBIT_THEME_JS_DIR . 'main.js', ['jquery'], false, true );
        wp_enqueue_script( 'nextbit-custom', NEXTBIT_THEME_JS_DIR . 'nextbit-custom.js', ['jquery'], false, true );
    }

    if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
        wp_enqueue_script( 'comment-reply' );
    }

}
add_action( 'wp_enqueue_scripts', 'nextbit_scripts' );

/*
Register Fonts
 */
function nextbit_fonts_url() {
    $font_url = '';
    /**
    * Translators: If there are characters in your language that are not supported
    * by chosen font(s), translate this to 'off'. Do not translate into your own language.
    */
    if ( 'off' !== _x( 'on', 'Google font: on or off', 'nextbit' ) ) {
        $font_url = 'https://fonts.googleapis.com/css2?family=Poppins:wght@100;300;400;500;600;700;800;900&display=swap';
        $font_url = 'https://fonts.googleapis.com/css2?family=Roboto:ital,wght@0,300;0,400;0,500;0,600;0,700;1,600&display=swap';
    }
    return $font_url;
}

// wp_body_open
if ( !function_exists( 'wp_body_open' ) ) {
    function wp_body_open() {
        do_action( 'wp_body_open' );
    }
}

/**
 * Implement the Custom Header feature.
 */
require NEXTBIT_THEME_INC . 'custom-header.php';

/**
 * Functions which enhance the theme by hooking into WordPress.
 */
require NEXTBIT_THEME_INC . 'nextbit-functions.php';

/**
 * Custom template helper function for this theme.
 */
require NEXTBIT_THEME_INC . 'nextbit-helper.php';


if (!defined('NEXTBIT_WOOCOMMERCE_ACTIVED')) {
    define('NEXTBIT_WOOCOMMERCE_ACTIVED', in_array('woocommerce/woocommerce.php', apply_filters('active_plugins', get_option('active_plugins'))));
}

define('NEXTBIT_COMPARE_ACTIVED', in_array('yith-woocommerce-compare/init.php', apply_filters('active_plugins', get_option('active_plugins'))));

define('NEXTBIT_QUICK_VIEW_ACTIVED', in_array('yith-woocommerce-quick-view/init.php', apply_filters('active_plugins', get_option('active_plugins'))));

define('NEXTBIT_WISHLIST_ACTIVED', in_array('yith-woocommerce-wishlist/init.php', apply_filters('active_plugins', get_option('active_plugins'))));

if (NEXTBIT_WOOCOMMERCE_ACTIVED) {
    require_once NEXTBIT_THEME_INC . 'nextbit-woocommerce.php';
}

/**
 * Load Jetpack compatibility file.
 */
if ( defined( 'JETPACK__VERSION' ) ) {
    require NEXTBIT_THEME_INC . 'jetpack.php';
}

/**
 * include nextbit functions file
 */
require_once NEXTBIT_THEME_INC . 'class-breadcrumb.php';
require_once NEXTBIT_THEME_INC . 'class-navwalker.php';
require_once NEXTBIT_THEME_INC . 'class-tgm-plugin-activation.php';
require_once NEXTBIT_THEME_INC . 'add-plugin.php';

/**
 * initialize kirki customizer class.
 */
include_once NEXTBIT_THEME_INC . 'kirki-customizer.php';
include_once NEXTBIT_THEME_INC . 'class-nextbit-kirki.php';

/**
 * Add a pingback url auto-discovery header for single posts, pages, or attachments.
 */
function nextbit_pingback_header() {
    if ( is_singular() && pings_open() ) {
        printf( '<link rel="pingback" href="%s">', esc_url( get_bloginfo( 'pingback_url' ) ) );
    }
}
add_action( 'wp_head', 'nextbit_pingback_header' );

/**
 *
 * comment section
 *
 */
add_filter( 'comment_form_default_fields', 'nextbit_comment_form_default_fields_func' );

function nextbit_comment_form_default_fields_func( $default ) {

    $default['author'] = '
    <div class="row clearfix">
        <div class="col-lg-4 col-md-4 col-sm-12 form-group">
            <label>'.esc_html__('Name', 'nextbit').'</label>
            <input type="text" name="author">
        </div>';

    $default['email'] = '
            <div class="col-lg-4 col-md-4 col-sm-12 form-group">
                <label>'.esc_html__('Email', 'nextbit').'</label>
                <input type="text" name="email">
            </div>

        ';
    // $default['url'] = '';
    $defaults['comment_field'] = '';

    $default['url'] = '
        <div class="col-lg-4 col-md-4 col-sm-12 form-group">
            <label>'.esc_html__('Website', 'nextbit').'</label>
            <input type="text" name="url">
    	</div>';

    $default['clients_commnet'] = '
    <div class="col-lg-12 col-md-12 col-sm-12 form-group">
        <label>'.esc_html__('Your comment here...', 'nextbit').'</label>
        <textarea id="comment" name="comment" cols="60" rows="6" aria-required="true"></textarea>
    </div>
    </div>';
    return $default;
}

add_action( 'comment_form_top', 'nextbit_add_comments_textarea' );
function nextbit_add_comments_textarea() {
    if ( !is_user_logged_in() ) {
        echo '<div class="row">
            <div class="col-lg-12 col-md-12 col-sm-12 form-group">
            <label>'.esc_html__('Your comment here...', 'nextbit').'</label>
            <textarea id="comment" name="comment" cols="60" rows="6" aria-required="true"></textarea>
        </div>
        </div>';
    }
}

add_filter( 'comment_form_defaults', 'nextbit_comment_form_defaults_func' );

function nextbit_comment_form_defaults_func( $info ) {
    if ( !is_user_logged_in() ) {
        $info['comment_field'] = '';
        $info['submit_field'] = '%1$s %2$s';
    } else {
        $info['comment_field'] = '<div class="row clearfix"><div class="col-lg-12 col-md-12 col-sm-12 form-group">
            <label>'.esc_html__('Your comment here...', 'nextbit').'</label>
            <textarea id="comment" name="comment" cols="30" rows="10"></textarea></div></div>';
        $info['submit_field'] = '%1$s %2$s';
    }

    $info['submit_button'] = '<div class="row clearfix"><div class="col-lg-12 col-md-12 col-sm-12 form-group"><button type="submit" class="theme-btn btn-style-four">
    <span class="txt">' . esc_html__( 'Post Comment', 'nextbit' ) . '<i class="lnr lnr-arrow-right"></i> </span></button></div></div>';

    $info['title_reply_before'] = '<div class="group-title"><h4>';
    $info['title_reply_after'] = '</h4></div><p> ' . esc_html__( 'Your email address will not be published. Required fields are marked', 'nextbit' ) . '* </p>';
    $info['comment_notes_before'] = '';

    return $info;
}

if ( !function_exists( 'nextbit_comment' ) ) {
    function nextbit_comment( $comment, $args, $depth ) {
        $GLOBAL['comment'] = $comment;
        extract( $args, EXTR_SKIP );
        $args['reply_text'] = '<button type="theme-btn reply-btn" class="reply_btn"> Reply</button>';
        $replayClass = 'comment-depth-' . esc_attr( $depth );
        ?>
        <li class="comment-box" id="comment-<?php comment_ID();?>">
            <div class="comment">
                <div class="author-thumb">
                    <?php print get_avatar( $comment, 90, null, null, ['class' => []] );?>
                </div>
                <div class="comment-text">
                    <div class="comment-info clearfix">
                        <strong><?php print get_comment_author_link();?></strong>
                        <div class="comment-time">
                            <?php comment_time( get_option( 'date_format' ) );?>
                        </div>
                    </div>
                    <div class="text"><?php comment_text();?></div>
                    <?php comment_reply_link( array_merge( $args, ['depth' => $depth, 'max_depth' => $args['max_depth']] ) );?>
                </div>
            </div>
        </li>
		<?php
    }
}

/**
 * shortcode supports for removing extra p, spance etc
 *
 */
add_filter( 'the_content', 'nextbit_shortcode_extra_content_remove' );
/**
 * Filters the content to remove any extra paragraph or break tags
 * caused by shortcodes.
 *
 * @since 1.0.0
 *
 * @param string $content  String of HTML content.
 * @return string $content Amended string of HTML content.
 */
function nextbit_shortcode_extra_content_remove( $content ) {

    $array = [
        '<p>['    => '[',
        ']</p>'   => ']',
        ']<br />' => ']',
    ];
    return strtr( $content, $array );

}

// nextbit_search_filter_form
if ( !function_exists( 'nextbit_search_filter_form' ) ) {
    function nextbit_search_filter_form( $form ) {

        $form = sprintf(
            '<form action="%s" method="get">
                <div class="form-group">
                    <input type="search" value="%s" required name="s" placeholder="%s">
                    <button type="submit"><span class="icon fa fa-search"></span></button>
                </div>
		    </form>',
            esc_url( home_url( '/' ) ),
            esc_attr( get_search_query() ),
            esc_html__( 'Search Here...', 'nextbit' )
        );

        return $form;
    }
    add_filter( 'get_search_form', 'nextbit_search_filter_form' );
}

add_action( 'admin_enqueue_scripts', 'nextbit_admin_custom_scripts' );

function nextbit_admin_custom_scripts() {
    wp_enqueue_media();
    wp_register_script( 'nextbit-admin-custom', get_template_directory_uri() . '/inc/js/admin_custom.js', ['jquery'], '', true );
    wp_enqueue_script( 'nextbit-admin-custom' );
}

// enable_rtl
function nextbit_enable_rtl() {
    if ( get_theme_mod( 'rtl_switch', false ) ) {
        return ' dir="rtl" ';
    } else {
        return '';
    }
}

// limit releted product show
if (NEXTBIT_WOOCOMMERCE_ACTIVED) {
    add_filter( 'woocommerce_output_related_products_args', 'nextbit_change_number_related_products', 9999 );

    function nextbit_change_number_related_products( $args ) {
        $args['posts_per_page'] = 2;
        $args['columns'] = 2;
        return $args;
    }
}