import { Principal } from '@dfinity/principal';
import { RequestId, RequestStatusResponseStatus } from '..';
import { PollStrategy } from './index';
export declare type Predicate<T> = (canisterId: Principal, requestId: RequestId, status: RequestStatusResponseStatus) => Promise<T>;
/**
 * A best practices polling strategy: wait 2 seconds before the first poll, then 1 second
 * with an exponential backoff factor of 1.2. Timeout after 5 minutes.
 */
export declare function defaultStrategy(): PollStrategy;
/**
 * Predicate that returns true once.
 */
export declare function once(): Predicate<boolean>;
/**
 * Delay the polling once.
 * @param condition A predicate that indicates when to delay.
 * @param timeInMsec The amount of time to delay.
 */
export declare function conditionalDelay(condition: Predicate<boolean>, timeInMsec: number): PollStrategy;
/**
 * Error out after a maximum number of polling has been done.
 * @param count The maximum attempts to poll.
 */
export declare function maxAttempts(count: number): PollStrategy;
/**
 * Throttle polling.
 * @param throttleInMsec Amount in millisecond to wait between each polling.
 */
export declare function throttle(throttleInMsec: number): PollStrategy;
/**
 * Reject a call after a certain amount of time.
 * @param timeInMsec Time in milliseconds before the polling should be rejected.
 */
export declare function timeout(timeInMsec: number): PollStrategy;
/**
 * A strategy that throttle, but using an exponential backoff strategy.
 * @param startingThrottleInMsec The throttle in milliseconds to start with.
 * @param backoffFactor The factor to multiple the throttle time between every poll. For
 *   example if using 2, the throttle will double between every run.
 */
export declare function backoff(startingThrottleInMsec: number, backoffFactor: number): PollStrategy;
/**
 * Chain multiple polling strategy. This _chains_ the strategies, so if you pass in,
 * say, two throttling strategy of 1 second, it will result in a throttle of 2 seconds.
 * @param strategies A strategy list to chain.
 */
export declare function chain(...strategies: PollStrategy[]): PollStrategy;
